-- scripts/AutomaticBeaconMini.lua (FS25)
-- Beacon ON/OFF controller that listens to ObjectChangeUtil.setObjectChanges without modifying other mods.

AutomaticBeaconMini = {}

local ORIG_setObjectChanges = nil

function AutomaticBeaconMini.prerequisitesPresent(_)
    return true
end

function AutomaticBeaconMini.registerEventListeners(vType)
    SpecializationUtil.registerEventListener(vType, "onLoad",     AutomaticBeaconMini)
    SpecializationUtil.registerEventListener(vType, "onDelete",   AutomaticBeaconMini)
    SpecializationUtil.registerEventListener(vType, "onUpdate",   AutomaticBeaconMini)
end

-- --------------- utils ---------------
local function dbg(spec, msg)
    if spec and spec.debug then
        print(string.format("AutomaticBeaconMini [%s]: %s", spec.vName or "vehicle", msg))
    end
end

local function apply(self, spec, want)
    if want == nil or want == spec.active then return end
    if self.setBeaconLightsVisibility ~= nil then
        self:setBeaconLightsVisibility(want)
        spec.active = want
        dbg(spec, "beacon -> " .. (want and "ON" or "OFF"))
    else
        dbg(spec, "WARNING: setBeaconLightsVisibility missing on this vehicle")
    end
end

local function anyActiveTriggers(spec)
    if not spec.triggers then return false end
    for _, on in pairs(spec.triggers) do if on then return true end end
    return false
end

-- --------------- specialization ---------------
function AutomaticBeaconMini:onLoad(_)
    local spec = {}
    self.spec_automaticBeaconMini = spec

    local xml, key = self.xmlFile, "vehicle.automaticBeaconMini"
    spec.enabled = true
    spec.debug   = false
    spec.vName   = self.typeName or "vehicle"
    spec.active  = false
    spec.triggers = {}          -- name -> bool
    spec.sentinels = {}         -- set of sentinel names to watch for
    spec._sentinelList = ""     -- raw string for logs

    if xml and xml:hasProperty(key) then
        spec.enabled = xml:getBool(key .. "#enabled", true)
        spec.debug   = xml:getInt(key .. "#debug", 0) == 1
        local raw    = xml:getString(key .. "#beaconSentinelNames") or ""
        spec._sentinelList = raw
        for tok in string.gmatch(raw, "%S+") do
            spec.sentinels[tok] = true
        end
    end

    -- one-time hook of ObjectChangeUtil.setObjectChanges
    if ORIG_setObjectChanges == nil and ObjectChangeUtil ~= nil and ObjectChangeUtil.setObjectChanges ~= nil then
        ORIG_setObjectChanges = ObjectChangeUtil.setObjectChanges
        ObjectChangeUtil.setObjectChanges = function(objectChanges, state, target, setMovingToolDirty)
            -- pre: nothing
            local ok, res = pcall(ORIG_setObjectChanges, objectChanges, state, target, setMovingToolDirty)
            -- post: inspect if target has our specialization
            if ok and target and target.spec_automaticBeaconMini then
                AutomaticBeaconMini._inspectObjectChangeCall(target, objectChanges, state)
            end
            if not ok then
                -- rethrow the original error to keep GIANTS behavior
                error(res)
            end
            return res
        end
        print("[AutomaticBeaconMini] Wrapped ObjectChangeUtil.setObjectChanges")
    end

    dbg(spec, ("loaded; sentinels=[%s]"):format(spec._sentinelList))
end

function AutomaticBeaconMini:onDelete()
    -- do not unhook global; other vehicles may need it
end

function AutomaticBeaconMini:onUpdate(dt)
    local spec = self.spec_automaticBeaconMini
    if not spec or not spec.enabled then return end
    -- keep enforcing aggregated state (handles user pressing Home key)
    local want = anyActiveTriggers(spec)
    apply(self, spec, want)
end

-- Called after the original ObjectChangeUtil.setObjectChanges runs
function AutomaticBeaconMini._inspectObjectChangeCall(vehicle, objectChanges, state)
    local spec = vehicle.spec_automaticBeaconMini
    if not (spec and spec.enabled) then return end
    if objectChanges == nil or spec.sentinels == nil then return end

    -- See if this objectChange group contains ANY sentinel node name
    local matchedName = nil
    for _, oc in ipairs(objectChanges) do
        local node = oc.node
        if node ~= nil then
            local nm = getName(node) or ""
            if spec.sentinels[nm] or spec.sentinels[tostring(nm)] then
                matchedName = nm
                break
            end
        end
    end

    if matchedName ~= nil then
        -- This group is one of our beacons triggers; mirror its state
        spec.triggers[matchedName] = (state == true)
        local any = anyActiveTriggers(spec)
        apply(vehicle, spec, any)
        dbg(spec, string.format("sentinel '%s' -> %s (any=%s)", tostring(matchedName), tostring(state), tostring(any)))
    end
end

-- Optional console test: abm.force 1|0 (force a virtual trigger)
if addConsoleCommand ~= nil then
    local function ABM_Force(val)
        local veh = g_currentMission and g_currentMission.controlledVehicle
        if veh == nil or veh.spec_automaticBeaconMini == nil then
            return "[ABM] No controlled vehicle or specialization missing."
        end
        local spec = veh.spec_automaticBeaconMini
        local on = tostring(val or "0") == "1"
        spec.triggers.__manual = on
        local any = anyActiveTriggers(spec)
        apply(veh, spec, any)
        return "[ABM] manual trigger -> " .. tostring(on)
    end
    pcall(function() addConsoleCommand("abm.force", "ABM_Force", "Toggle a manual trigger: abm.force 1|0") end)
end
